<?php

use Illuminate\Support\Facades\Route;

use App\Http\Controllers\DashboardController;
use App\Http\Controllers\MemberController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\AccountTypeController;
use App\Http\Controllers\AccountController;
use App\Http\Controllers\TransactionController;
use App\Http\Controllers\LoanTypeController;
use App\Http\Controllers\LoanController;
use App\Http\Controllers\GroupLoanController;
use App\Http\Controllers\GroupLoanRepaymentController;

// ------------------------------------------------------------
// PUBLIC
// ------------------------------------------------------------
Route::get('/', function () {
    return view('landing');
});

// Login / Register
require __DIR__.'/auth.php';

// Shared Dashboard
Route::middleware(['auth', 'verified'])->get('/dashboard', 
    [DashboardController::class, 'index'])->name('dashboard');



// ------------------------------------------------------------
// MEMBER ROUTES (Non-Admin) — Personal Loans ONLY
// ------------------------------------------------------------
Route::middleware(['auth'])->prefix('member')->group(function () {

    Route::get('/dashboard', 
        [MemberController::class, 'dashboard'])
        ->name('member.dashboard');

    // Loan Application
    Route::get('/loans/apply', 
        [LoanController::class, 'apply'])
        ->name('member.loans.apply');

    Route::post('/loans/apply', 
        [LoanController::class, 'store'])
        ->name('member.loans.submit');

    // View personal loans
    Route::get('/loans', 
        [LoanController::class, 'memberLoans'])
        ->name('member.loans.index');

    Route::get('/loans/{loan}', 
        [LoanController::class, 'show'])
        ->name('member.loans.show');

    // Repay
    Route::post('/loans/{loan}/repay', 
        [LoanController::class, 'repay'])
        ->name('member.loans.repay');
});



// ------------------------------------------------------------
// ADMIN ROUTES
// ------------------------------------------------------------
Route::middleware(['auth', 'admin'])
    ->prefix('admin')
    ->group(function () {

    // -----------------------------
    // ADMIN HOME
    // -----------------------------
    Route::get('/', [App\Http\Controllers\AdminController::class, 'index'])
        ->name('admin.dashboard');



    // -----------------------------
    // USERS
    // -----------------------------
    Route::resource('users', UserController::class);

    Route::post('/users/{user}/status/{status}', 
        [UserController::class, 'updateStatus'])
        ->name('users.status');

    Route::get('/users/{user}/statement', 
        [TransactionController::class, 'memberStatementPdf'])
        ->name('admin.users.statement');



    // -----------------------------
    // ACCOUNT TYPES
    // -----------------------------
    Route::resource('account-types', AccountTypeController::class, [
        'as' => 'admin'
    ]);

    Route::post('/account-types/{type}/toggle-status', 
        [AccountTypeController::class, 'toggleStatus'])
        ->name('admin.account_types.toggle_status');



    // -----------------------------
    // ACCOUNTS
    // -----------------------------
    Route::get('/accounts', 
        [AccountController::class, 'index'])->name('admin.accounts.index');

    Route::get('/accounts/create', 
        [AccountController::class, 'create'])->name('admin.accounts.create');

    Route::post('/accounts', 
        [AccountController::class, 'store'])->name('admin.accounts.store');

    Route::get('/accounts/{account}/edit', 
        [AccountController::class, 'edit'])->name('admin.accounts.edit');

    Route::put('/accounts/{account}', 
        [AccountController::class, 'update'])->name('admin.accounts.update');

    Route::post('/accounts/{account}/toggle-status', 
        [AccountController::class, 'toggleStatus'])->name('admin.accounts.toggle_status');

    Route::delete('/accounts/{account}', 
        [AccountController::class, 'destroy'])->name('admin.accounts.destroy');

    Route::get('/accounts/{account}', 
        [AccountController::class, 'show'])->name('admin.accounts.show');



    // -----------------------------
    // TRANSACTIONS
    // -----------------------------
    Route::get('/accounts/{account}/transactions', 
        [TransactionController::class, 'index'])
        ->name('admin.accounts.transactions');

    Route::post('/accounts/{account}/deposit', 
        [TransactionController::class, 'deposit'])
        ->name('admin.accounts.deposit');

    Route::post('/accounts/{account}/withdraw', 
        [TransactionController::class, 'withdraw'])
        ->name('admin.accounts.withdraw');

    // Receipt
    Route::get('/transactions/{id}/receipt', 
        [TransactionController::class, 'receipt'])
        ->name('admin.transactions.receipt');

    // Global Transactions List
    Route::get('/transactions', 
        [TransactionController::class, 'index'])
        ->name('admin.transactions.index');

    Route::get('/transactions/export/csv', 
        [TransactionController::class, 'exportCsv'])
        ->name('admin.transactions.export.csv');

    Route::get('/transactions/export/excel', 
        [TransactionController::class, 'exportExcel'])
        ->name('admin.transactions.export.excel');

    Route::get('/transactions/export/financial-csv', 
        [TransactionController::class, 'exportFinancialCsv'])
        ->name('admin.transactions.export.financial_csv');



    // -----------------------------
    // LOAN TYPES
    // -----------------------------
    Route::resource('loan_types', LoanTypeController::class, [
        'as' => 'admin'
    ]);



    // ------------------------------------------------------------
    // PERSONAL LOANS (ADMIN)
    // ------------------------------------------------------------
    Route::get('/loans', 
        [LoanController::class, 'adminIndex'])->name('admin.loans.index');

    Route::get('/loans/{loan}', 
        [LoanController::class, 'show'])->name('admin.loans.show');

    Route::post('/loans/{loan}/approve', 
        [LoanController::class, 'approve'])->name('admin.loans.approve');

    Route::post('/loans/{loan}/reject', 
        [LoanController::class, 'reject'])->name('admin.loans.reject');

    Route::post('/loans/{loan}/disburse', 
        [LoanController::class, 'disburse'])->name('admin.loans.disburse');

    Route::post('/loans/{loan}/repay', 
        [LoanController::class, 'repay'])->name('admin.loans.repay');



    // ------------------------------------------------------------
    // GROUP LOANS (ADMIN)
    // ------------------------------------------------------------
    Route::get('/group-loans', 
        [GroupLoanController::class, 'index'])
        ->name('admin.group_loans.index');

    Route::get('/group-loans/{loan}', 
        [GroupLoanController::class, 'show'])
        ->name('admin.group_loans.show');

    Route::get('/groups/{group}/loans/create', 
        [GroupLoanController::class, 'create'])
        ->name('admin.group_loans.create');

    Route::post('/groups/{group}/loans', 
        [GroupLoanController::class, 'store'])
        ->name('admin.group_loans.store');

    Route::post('/group-loans/{loan}/approve', 
        [GroupLoanController::class, 'approve'])
        ->name('admin.group_loans.approve');

    Route::post('/group-loans/{loan}/disburse', 
        [GroupLoanController::class, 'disburse'])
        ->name('admin.group_loans.disburse');

    Route::post('/group-loans/{loan}/repayments', 
        [GroupLoanRepaymentController::class, 'store'])
        ->name('admin.group_loan_repayments.store');

    Route::get('/repayments/{repayment}/receipt', 
        [GroupLoanRepaymentController::class, 'receipt'])
        ->name('admin.group_loan_repayments.receipt');
});
