<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Str;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * Mass assignable attributes.
     * Note: password intentionally removed from fillable to avoid mass-assignment issues.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'phone',
        'address',
        'national_id',
        'role',
        'status',
        'member_id',
        'photo',
        'password', ];

    /**
     * Hidden attributes for arrays.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Attribute casting.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        // password hashing handled in controllers or mutators
    ];

    /**
     * Boot method to hook into model events.
     * We ensure every user has a unique member_id on creation.
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($user) {
            // If member_id is empty, generate a unique one
            if (empty($user->member_id)) {
                $user->member_id = self::generateUniqueMemberId();
            }

            // Optionally set default role/status if missing
            if (empty($user->role)) {
                $user->role = 'member';
            }
            if (empty($user->status)) {
                $user->status = 'active';
            }
        });
    }

    /**
     * Generate a unique member id.
     * Format: MBR-XXXXXX (uppercase letters+digits)
     *
     * @return string
     */
    protected static function generateUniqueMemberId()
    {
        // change prefix if you prefer 'HCS-' or other
        $prefix = 'MBR-';

        do {
            // 6 char alphanumeric (avoid confusing chars if you want)
            $random = strtoupper(Str::random(6));
            $candidate = $prefix . $random;
        } while (self::where('member_id', $candidate)->exists());

        return $candidate;
    }

    /**
     * Relations
     */
    public function accounts()
    {
        return $this->hasMany(\App\Models\Account::class);
    }

    public function groupMembers()
    {
        return $this->hasMany(\App\Models\GroupMember::class);
    }


    /**
     * Role check helpers
     */
    public function isAdmin()
    {
        return $this->role === 'admin';
    }

    public function isMember()
    {
        return $this->role === 'member';
    }

    public function isAccountant()
    {
        return $this->role === 'accountant';
    }

    public function isLoanOfficer()
    {
        return $this->role === 'loan_officer';
    }


    
}
