<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\Loan;
use App\Models\IndividualLoanRepayment;
use App\Models\Transaction;
use Illuminate\Http\Request;

class MemberController extends Controller
{
    public function dashboard()
    {
        $user = auth()->user();

        //------------------------------------------
        // 1. Fetch Member Accounts
        //------------------------------------------
        $accounts = Account::where('user_id', $user->id)->get();

        //------------------------------------------
        // 2. Recent Transactions (last 10)
        //------------------------------------------
        $recentTransactions = Transaction::whereHas('account', function($q) use ($user) {
                $q->where('user_id', $user->id);
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        //------------------------------------------
        // 3. Total Deposits & Withdrawals
        //------------------------------------------
        $deposits = Transaction::whereHas('account', function($q) use ($user) {
                $q->where('user_id', $user->id);
            })
            ->where('type', 'deposit')
            ->sum('amount');

        $withdrawals = Transaction::whereHas('account', function($q) use ($user) {
                $q->where('user_id', $user->id);
            })
            ->where('type', 'withdrawal')
            ->sum('amount');

        //------------------------------------------
        // 4. Active Loan (approved or active)
        //------------------------------------------
        $activeLoan = Loan::where('user_id', $user->id)
            ->whereIn('status', ['approved', 'active'])
            ->with(['loanType', 'repayments']) // eager load for performance
            ->first();

        //------------------------------------------
        // 5. Next Pending Repayment
        //------------------------------------------
 // Next repayment from amortization schedule
$nextRepayment = null;

if ($activeLoan && is_array($activeLoan->amortization_schedule)) {
    $nextRepayment = collect($activeLoan->amortization_schedule)
        ->where('balance', '>', 0)  // only future payments
        ->sortBy('period')
        ->first();
}

$loanSchedule = $activeLoan && is_array($activeLoan->amortization_schedule)
    ? collect($activeLoan->amortization_schedule)->sortBy('period')
    : collect();



        //------------------------------------------
        // 6. Return View
        //------------------------------------------
      return view('member.dashboard', compact(
    'user',
    'accounts',
    'recentTransactions',
    'activeLoan',
    'nextRepayment',
    'loanSchedule',
    'deposits',
    'withdrawals'
));

    }

//insights page
public function insights()
{
    $user = auth()->user();

    // Get all transactions
    $transactions = Transaction::whereHas('account', function($q) use ($user) {
        $q->where('user_id', $user->id);
    })->orderBy('created_at')->get();

    // GROUP MONTHLY SAVINGS
    $monthlyDeposits = $transactions->where('type', 'deposit')
        ->groupBy(fn($t) => $t->created_at->format('Y-m'))
        ->map(fn($group) => $group->sum('amount'));

    $monthlyWithdrawals = $transactions->where('type', 'withdrawal')
        ->groupBy(fn($t) => $t->created_at->format('Y-m'))
        ->map(fn($group) => $group->sum('amount'));

    // SUM TOTALS for PIE CHART
    $totalDeposits = $transactions->where('type', 'deposit')->sum('amount');
    $totalWithdrawals = $transactions->where('type', 'withdrawal')->sum('amount');

    // ACTIVE LOAN
    $loan = Loan::where('user_id', $user->id)
        ->whereIn('status', ['approved', 'active', 'closed'])
        ->first();

    $loanSchedule = $loan && is_array($loan->amortization_schedule)
        ? collect($loan->amortization_schedule)
        : collect();

    return view('member.insights', compact(
        'user',
        'monthlyDeposits',
        'monthlyWithdrawals',
        'totalDeposits',
        'totalWithdrawals',
        'loan',
        'loanSchedule'
    ));
}


public function loanStatement(Loan $loan)
{
    $this->authorizeMemberOrAdmin($loan);

    $repayments = $loan->repayments()->orderBy('created_at')->get();

    $pdf = \PDF::loadView('pdf.loan-statement', [
        'loan' => $loan,
        'repayments' => $repayments,
        'schedule' => collect($loan->amortization_schedule)
    ]);

    return $pdf->download("Loan-Statement-{$loan->id}.pdf");
}
// member statement
public function statement(Request $request)
{
    $user = auth()->user();

    // Member accounts
    $accounts = $user->accounts;
    $accountId = $request->account_id ?? ($accounts->first()->id ?? null);

    if (!$accountId) {
        return view('member.statement', [
            'accounts' => $accounts,
            'transactions' => collect(),
            'accountId' => null,
            'openingBalance' => 0,
            'closingBalance' => 0,
            'totalCredit' => 0,
            'totalDebit' => 0,
            'statementStartDate' => now(),
        ]);
    }

    // Fetch account
    $account = Account::find($accountId);

    // --------------------------------
    // 1. Fetch ACCOUNT transactions only
    // --------------------------------
    $transactions = Transaction::where('account_id', $accountId)
        ->orderBy('created_at', 'asc')
        ->get()
        ->map(function ($tx) {
            return [
                'date' => $tx->created_at,
                'description' => $tx->description ?? ucfirst($tx->type),
                'credit' => $tx->type === 'deposit' ? $tx->amount : null,
                'debit'  => $tx->type === 'withdrawal' ? $tx->amount : null,
                'balance' => $tx->balance_after,
            ];
        });

    // --------------------------------
    // 2. Summary Calculations
    // --------------------------------
    $statementStartDate = $transactions->first()['date'] ?? now();

    $totalCredit = $transactions->sum(fn($t) => $t['credit'] ?? 0);
    $totalDebit  = $transactions->sum(fn($t) => $t['debit'] ?? 0);

    // Closing balance is the last account balance
    $closingBalance = Transaction::where('account_id', $accountId)
        ->orderBy('created_at', 'desc')
        ->value('balance_after') ?? 0;

    // Opening balance = closing balance - net movement
    $openingBalance = $closingBalance - ($totalCredit - $totalDebit);

    return view('member.statement', compact(
        'accounts',
        'transactions',
        'accountId',
        'openingBalance',
        'closingBalance',
        'totalCredit',
        'totalDebit',
        'statementStartDate',
        'account'
    ));
}


public function statementPdf(Request $request)
{
    $user = auth()->user();
    $accounts = $user->accounts;

    $accountId = $request->account_id ?? ($accounts->first()->id ?? null);
    $account = Account::findOrFail($accountId);

    $transactions = Transaction::where('account_id', $accountId)
        ->orderBy('created_at', 'asc')
        ->get()
        ->map(fn($tx) => [
            'date' => $tx->created_at->format('Y-m-d H:i:s'),
            'description' => $tx->description ?? ucfirst($tx->type),
            'credit' => $tx->type === 'deposit' ? $tx->amount : null,
            'debit'  => $tx->type === 'withdrawal' ? $tx->amount : null,
            'balance' => $tx->balance_after,
        ]);

    $closingBalance = $transactions->last()['balance'] ?? 0;
    $openingBalance = $transactions->first()['balance'] ?? 0;

    $pdf = \PDF::loadView('pdf.member-statement', [
        'account' => $account,
        'transactions' => $transactions,
        'openingBalance' => $openingBalance,
        'closingBalance' => $closingBalance,
    ]);

    return $pdf->download("Member-Statement-{$account->account_number}.pdf");
}



}
