<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Income;
use App\Models\IncomeType;
use App\Models\User;
use App\Models\Account;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class IncomeController extends Controller
{
    // Auto receipt number
    protected function receipt()
    {
        return 'RCPT-' . strtoupper(Str::random(8));
    }

    /**
     * UPDATED INDEX WITH FILTERS
     */
    public function index(Request $request)
    {
        $query = Income::with(['type', 'admin', 'member'])
            ->orderBy('income_date', 'desc');

        // --- Search (description, amount, member name, member ID)
        if ($request->search) {
            $search = $request->search;

            $query->where(function ($q) use ($search) {
                $q->where('description', 'like', "%$search%")
                  ->orWhere('amount', 'like', "%$search%")
                  ->orWhereHas('member', function ($m) use ($search) {
                      $m->where('name', 'like', "%$search%")
                        ->orWhere('member_id', 'like', "%$search%");
                  });
            });
        }

        // --- Filter: Income Type
        if ($request->income_type_id) {
            $query->where('income_type_id', $request->income_type_id);
        }

        // --- Filter: Payment Method
        if ($request->payment_method) {
            $query->where('payment_method', $request->payment_method);
        }

        // --- Filter: Member
        if ($request->member_id) {
            $query->where('member_id', $request->member_id);
        }

        // --- Filter: Date From
        if ($request->date_from) {
            $query->whereDate('income_date', '>=', $request->date_from);
        }

        // --- Filter: Date To
        if ($request->date_to) {
            $query->whereDate('income_date', '<=', $request->date_to);
        }

        $incomes = $query->paginate(20)->withQueryString(); // keeps filters when paginating
        $incomeTypes = IncomeType::all();
        $members = User::where('role', 'member')->get();

        return view('admin.incomes.index', compact('incomes', 'incomeTypes', 'members'));
    }

    public function create()
    {
        $incomeTypes = IncomeType::all();
        $members     = User::where('role', 'member')->get();
        $accounts    = Account::all();

        return view('admin.incomes.create', compact('incomeTypes', 'members', 'accounts'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'income_type_id' => 'required|exists:income_types,id',
            'member_id'      => 'nullable|exists:users,id',
            'account_id'     => 'required|exists:accounts,id',
            'payment_method' => 'required|in:cash,mobile_money,bank',
            'amount'         => 'required|numeric|min:1',
            'income_date'    => 'required|date',
            'description'    => 'nullable|max:500',
        ]);

        Income::create([
            'receipt_no'     => $this->receipt(),
            'income_type_id' => $request->income_type_id,
            'member_id'      => $request->member_id,
            'account_id'     => $request->account_id,
            'payment_method' => $request->payment_method,
            'amount'         => $request->amount,
            'income_date'    => $request->income_date,
            'description'    => $request->description,
            'recorded_by'    => auth()->id(),
        ]);

        return redirect()->route('admin.incomes.index')
                         ->with('success', 'Income recorded successfully.');
    }

    public function edit(Income $income)
    {
        $incomeTypes = IncomeType::all();
        $members     = User::where('role', 'member')->get();
        $accounts    = Account::all();

        return view('admin.incomes.edit', compact('income', 'incomeTypes', 'members', 'accounts'));
    }

    public function update(Request $request, Income $income)
    {
        $request->validate([
            'income_type_id' => 'required|exists:income_types,id',
            'member_id'      => 'nullable|exists:users,id',
            'account_id'     => 'required|exists:accounts,id',
            'payment_method' => 'required|in:cash,mobile_money,bank',
            'amount'         => 'required|numeric|min:1',
            'income_date'    => 'required|date',
            'description'    => 'nullable|max:500',
        ]);

        $income->update($request->all());

        return redirect()->route('admin.incomes.index')
                         ->with('success', 'Income updated.');
    }

    public function destroy(Income $income)
    {
        $income->delete();
        return back()->with('success', 'Income deleted.');
    }


    public function exportCsv(Request $request)
{
    // Apply same filters as index
    $query = Income::with('type', 'member', 'admin');

    if ($request->search) {
        $query->where(function ($q) use ($request) {
            $q->where('description', 'like', "%{$request->search}%")
              ->orWhere('amount', 'like', "%{$request->search}%")
              ->orWhereHas('member', function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('member_id', 'like', "%{$request->search}%");
              });
        });
    }

    if ($request->income_type_id) {
        $query->where('income_type_id', $request->income_type_id);
    }

    if ($request->payment_method) {
        $query->where('payment_method', $request->payment_method);
    }

    if ($request->member_id) {
        $query->where('member_id', $request->member_id);
    }

    if ($request->date_from) {
        $query->whereDate('income_date', '>=', $request->date_from);
    }

    if ($request->date_to) {
        $query->whereDate('income_date', '<=', $request->date_to);
    }

    $incomes = $query->orderBy('income_date', 'desc')->get();

    // Build CSV output
    $filename = "income_report_" . now()->format('Ymd_His') . ".csv";

    $headers = [
        "Content-type"        => "text/csv",
        "Content-Disposition" => "attachment; filename=$filename",
        "Pragma"              => "no-cache",
        "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
        "Expires"             => "0"
    ];

    $columns = ["Receipt", "Type", "Member", "Amount", "Payment", "Date", "Recorded By"];

    $callback = function () use ($incomes, $columns) {
        $file = fopen('php://output', 'w');
        fputcsv($file, $columns);

        foreach ($incomes as $inc) {
            fputcsv($file, [
                $inc->receipt_no,
                $inc->type->name,
                $inc->member->name ?? '—',
                $inc->amount,
                $inc->payment_method,
                $inc->income_date,
                $inc->admin->name,
            ]);
        }

        fclose($file);
    };

    return response()->stream($callback, 200, $headers);
}


public function exportPdf(Request $request)
{
    $query = Income::with('type', 'member', 'admin');

    // Apply filters as in CSV
    if ($request->search) {
        $query->where(function ($q) use ($request) {
            $q->where('description', 'like', "%{$request->search}%")
              ->orWhere('amount', 'like', "%{$request->search}%")
              ->orWhereHas('member', function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('member_id', 'like', "%{$request->search}%");
              });
        });
    }

    if ($request->income_type_id) {
        $query->where('income_type_id', $request->income_type_id);
    }

    if ($request->payment_method) {
        $query->where('payment_method', $request->payment_method);
    }

    if ($request->member_id) {
        $query->where('member_id', $request->member_id);
    }

    if ($request->date_from) {
        $query->whereDate('income_date', '>=', $request->date_from);
    }

    if ($request->date_to) {
        $query->whereDate('income_date', '<=', $request->date_to);
    }

    $incomes = $query->orderBy('income_date', 'desc')->get();

    $pdf = \PDF::loadView('admin.incomes.pdf', [
        'incomes' => $incomes,
        'generated_at' => now()->format('d M Y - H:i'),
    ])->setPaper('A4', 'portrait');

    return $pdf->download('Income_Report_' . now()->format('Ymd_His') . '.pdf');
}




public function dashboard()
{
    $today = Carbon::today();
    $startOfMonth = Carbon::now()->startOfMonth();

    // Top KPI cards
    $totalIncome      = Income::sum('amount');
    $totalThisMonth   = Income::whereBetween('income_date', [$startOfMonth, now()])->sum('amount');
    $totalToday       = Income::whereDate('income_date', $today)->sum('amount');
    $totalTransactions = Income::count();

    // Income by type (for pie / bar chart)
    $incomeByType = Income::selectRaw('income_type_id, SUM(amount) as total')
        ->with('type')
        ->groupBy('income_type_id')
        ->get();

    // Monthly trend (last 6 months)
    $sixMonthsAgo = Carbon::now()->subMonths(5)->startOfMonth();

    $monthlyIncome = Income::selectRaw('DATE_FORMAT(income_date, "%Y-%m") as month, SUM(amount) as total')
        ->where('income_date', '>=', $sixMonthsAgo)
        ->groupBy('month')
        ->orderBy('month')
        ->get();

    // Payment method breakdown


    // Recent incomes
    $recentIncomes = Income::with('type', 'member', 'admin')
        ->orderBy('income_date', 'desc')
        ->limit(10)
        ->get();

    // Top contributing members
    $topMembers = Income::selectRaw('member_id, SUM(amount) as total')
        ->whereNotNull('member_id')
        ->with('member')
        ->groupBy('member_id')
        ->orderByDesc('total')
        ->limit(5)
        ->get();

return view('admin.incomes.dashboard', compact(
    'totalIncome',
    'totalThisMonth',
    'totalToday',
    'totalTransactions',
    'incomeByType',
    'monthlyIncome',
    'recentIncomes',
    'topMembers',

));


    // Monthly expenses (last 6 months)
$monthlyExpenses = \App\Models\Expense::selectRaw('DATE_FORMAT(expense_date, "%Y-%m") as month, SUM(amount) as total')
    ->where('expense_date', '>=', $sixMonthsAgo)
    ->groupBy('month')
    ->orderBy('month')
    ->get();

// Combine income + expenses per month (matching months)
$comparison = [];
foreach ($monthlyIncome as $row) {
    $month = $row->month;
    $comparison[$month]['income'] = $row->total;
}

foreach ($monthlyExpenses as $row) {
    $month = $row->month;
    $comparison[$month]['expense'] = $row->total;
}

// Normalize empty values
foreach ($comparison as $month => $values) {
    $comparison[$month]['income']  = $values['income']  ?? 0;
    $comparison[$month]['expense'] = $values['expense'] ?? 0;
    $comparison[$month]['net']     = $comparison[$month]['income'] - $comparison[$month]['expense'];
}

$comparisonData = collect($comparison);

}


}
