<?php

namespace App\Http\Controllers;

use App\Models\Account;
use App\Models\User;
use App\Models\AccountType;
use Illuminate\Http\Request;
use App\Services\AccountNumberGenerator;
use App\Exports\AccountsExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Transaction;

class AccountController extends Controller
{
    public function index(Request $request)
    {
        $query = Account::with('user', 'type');

        if ($request->filled('search')) {
            $search = $request->search;

            $query->where('account_number', 'LIKE', "%$search%")
                  ->orWhereHas('user', function($q) use ($search) {
                      $q->where('name', 'LIKE', "%$search%")
                        ->orWhere('email', 'LIKE', "%$search%");
                  });
        }

        if ($request->filled('type')) {
            $query->where('account_type_id', $request->type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $accounts = $query->paginate(10)->withQueryString();
        $types = AccountType::all();

        return view('admin.accounts.index', compact('accounts', 'types'));
    }

    public function create()
    {
        $members = User::where('role', 'member')->get();
        $types = AccountType::where('status', 'active')->get();

        $generated = AccountNumberGenerator::generate();

        return view('admin.accounts.create', compact('members', 'types', 'generated'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'user_id'          => 'required|exists:users,id',
            'account_type_id'  => 'required|exists:account_types,id',
            'notes'            => 'nullable|string|max:500',   // ✅ NEW
        ]);

        Account::create([
            'account_number'   => AccountNumberGenerator::generate(),
            'user_id'          => $request->user_id,
            'account_type_id'  => $request->account_type_id,
            'balance'          => 0,
            'status'           => 'active',
            'notes'            => $request->notes,             // ✅ NEW
        ]);

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Account created successfully.');
    }

    public function edit(Account $account)
    {
        $members = User::where('role', 'member')->get();
        $types = AccountType::where('status', 'active')->get();

        return view('admin.accounts.edit', compact('account', 'members', 'types'));
    }

    public function update(Request $request, Account $account)
    {
        $request->validate([
            'user_id'         => 'required|exists:users,id',
            'account_type_id' => 'required|exists:account_types,id',
            'status'          => 'required',
            'notes'           => 'nullable|string|max:500',     // ✅ NEW
        ]);

        $account->update([
            'user_id'         => $request->user_id,
            'account_type_id' => $request->account_type_id,
            'status'          => $request->status,
            'notes'           => $request->notes,               // ✅ NEW
        ]);

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Account updated successfully.');
    }

    public function toggleStatus(Account $account)
    {
        $account->status = $account->status === 'active' ? 'inactive' : 'active';
        $account->save();

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Status updated.');
    }

    public function destroy(Account $account)
    {
        $account->delete();

        return redirect()->route('admin.accounts.index')
            ->with('success', 'Account deleted.');
    }

    public function exportCsv()
    {
        $filename = 'accounts_export_' . date('Y-m-d_H-i-s') . '.csv';

        $accounts = Account::with('user', 'type')->get();

        $headers = [
            "Content-type"        => "text/csv",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        ];

        $columns = [
            'Account Number',
            'Member Name',
            'Member Email',
            'Account Type',
            'Balance (UGX)',
            'Status',
            'Notes',                   // ✅ NEW
            'Created At'
        ];

        $callback = function() use ($accounts, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($accounts as $acc) {
                fputcsv($file, [
                    $acc->account_number,
                    $acc->user->name,
                    $acc->user->email,
                    $acc->type->name,
                    number_format($acc->balance),
                    ucfirst($acc->status),
                    $acc->notes,             // ✅ NEW
                    $acc->created_at->format('Y-m-d H:i'),
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function exportExcel(Request $request)
    {
        $filters = [
            'search' => $request->search,
            'type'   => $request->type,
            'status' => $request->status,
        ];

        $filename = 'accounts_export_' . date('Y-m-d_H-i-s') . '.xlsx';

        return Excel::download(new AccountsExport($filters), $filename);
    }

    public function show(Account $account)
    {
        $transactions = Transaction::where('account_id', $account->id)
                                   ->orderBy('created_at', 'desc')
                                   ->paginate(10);

        return view('admin.accounts.show', compact('account', 'transactions'));
    }
}
